### A Pluto.jl notebook ###
# v0.20.21

using Markdown
using InteractiveUtils

# This Pluto notebook uses @bind for interactivity. When running this notebook outside of Pluto, the following 'mock version' of @bind gives bound variables a default value (instead of an error).
macro bind(def, element)
    #! format: off
    return quote
        local iv = try Base.loaded_modules[Base.PkgId(Base.UUID("6e696c72-6542-2067-7265-42206c756150"), "AbstractPlutoDingetjes")].Bonds.initial_value catch; b -> missing; end
        local el = $(esc(element))
        global $(esc(def)) = Core.applicable(Base.get, el) ? Base.get(el) : iv(el)
        el
    end
    #! format: on
end

# ╔═╡ b72215bb-8e3a-46e2-8817-271befe1178c
begin
	using PlutoUI, PlutoPlotly, PlotlyKaleido, Dates,  CSV, LaTeXStrings
	using DataFrames,  LinearAlgebra, StatsBase, Statistics, ShiftedArrays, Latexify
	force_pluto_mathjax_local(true)
	PlotlyKaleido.start(;mathjax = true)
end

# ╔═╡ 9d1ca4cc-688c-4693-9311-25b33feca957
html"""
<picture>
  <source srcset="https://gci.iscte-iul.pt/2022/identidade/files/brand_v2/escolas/rgb/ibs/en/horizontal/negative/rgb_ibs_en_horizontal_negative.png" media="(prefers-color-scheme: dark)" width="240px!important" style="float:right;">
  <img src="https://gci.iscte-iul.pt/2022/identidade/files/brand_v2/escolas/rgb/ibs/en/horizontal/main/rgb_ibs_en_horizontal_main.png" width="240px" style="float:right;">
</picture>
"""

# ╔═╡ 0b869e70-3717-11ec-202c-6b94dac7daf3
md"""
# Recap: Basic Statistics & Dynamic Processes

**02 February 2026**
"""

# ╔═╡ b326715f-c37f-44d2-92f9-6264368a18b2
md"""
##### Vivaldo Mendes
"""

# ╔═╡ 4fe446fc-10cf-4439-ae07-e0dad3a54de0
md"""
---
"""

# ╔═╡ 554b2cc5-99e1-4eda-b080-67ab2eddac16
md"""
### Packages used in this notebook
"""

# ╔═╡ 4d91f31e-06ce-4124-8a10-6619010c9476
md"""
## 1. Basic statistical concepts
"""

# ╔═╡ 5e8a5e83-9cc0-4bb6-86d2-4ec5f3d19b99
md"""
This is how most economic data looks like:
"""

# ╔═╡ b5106532-729d-46b5-b0fe-886c4349ee11
md"""
This is how random shocks look like:
"""

# ╔═╡ c81b7a4a-21fb-4643-ba05-7b744d29e70a
plot(randn(300), 
	mode="markers+lines",
	marker_symbol="circle", marker_size="5",line_width= 0.3,
    marker_color = "DarkRed", 
	Layout(height = 450, 
		title_text ="Two hundred random normally distributed observations",
		titlefont_size = "18"
        )
	)

# ╔═╡ 22f0e100-55e6-4849-91cb-c22fc334b5cc


# ╔═╡ 63610272-6b65-41f6-a71d-eb3c1327e4b6
md"""
#### The sample mean
"""

# ╔═╡ 30f88ccb-735b-4842-8598-751d985a7ff0
md"""
For a given real-valued array $X$, the sample *mean* is given by

$$\overline{x}=\frac{x_{1}+x_{2}+\ldots+x_{n}}{n}=\frac{\sum_{i=1}^{n} x_{i}}{n} \tag{1}$$

where $n$ is the length of the input, $x_i$ are the observations in $X$, and $\overline{x}$ is the sample mean.
"""

# ╔═╡ 116cc0c2-2674-4c05-948c-bc3f1915e30c
md"""
#### Sample variance and standard deviation
"""

# ╔═╡ eaf08952-9ce9-4c80-a708-52487825e13f
md"""
The sample *variance* and the sample *standard deviation* are given by, respectively:

$$s_x^{2}=\operatorname{Var}(x)=\frac{\sum_{i=1}^{n}\left(x_{i}-\bar{x}\right)^{2}}{n-1} \tag{2}$$

and

$$s_x=\sqrt{\frac{\sum_{i=1}^{n}\left(x_{i}-\bar{x}\right)^{2}}{n-1}} \tag{3}$$
"""

# ╔═╡ b7113a0b-6370-461f-ad47-d5e16c81c6cd
md"""
#### The Auto-Correlation function
"""

# ╔═╡ b8b07199-9deb-4108-95ea-12a2bcd5069c
md"""
To compute the covariance between the vectors $x$ and $y$, we can use the Pearson's **correlation coefficient**. When applied to a sample, the former is commonly represented by $r_{xy}$ and may be referred to as the sample correlation coefficient. We can obtain a formula for $r_{xy}$ by substituting estimates of the covariances and variances based on a sample into the formula above. Given paired data $\left\{\left(x_{1}, y_{1}\right), \ldots,\left(x_{n}, y_{n}\right)\right\}$ consisting of $n$ pairs, $r_{xy}$ is defined as:

$r_{xy}=\frac{1}{n-1} \sum_{i=1}^{n}\left(x_{i}-\bar{x}\right)\left(y_{i}-\bar{y}\right) \tag{4}$

where:

-- $n$ is sample size

-- $x_{i}, y_{i}$ are the individual sample points indexed with $i$

-- $\bar{x}=\frac{1}{n} \sum_{i=1}^{n} x_{i}$ (the sample mean); and analogously for $\bar{y}$

The **autocorrelation function** with $k$ lags of vector $y$ is defined as

$$r_{y}(k)=\frac{\sum_{i=1}^{n-k}\left(y_{i}-\bar{y}\right)\left(y_{i+k}-\bar{y}\right)}{\sum_{i=1}^{n}\left(y_{i}-\bar{y}\right)^{2}} \tag{5}$$
"""

# ╔═╡ 6d03748d-00e0-4c18-a8bc-d432a72714d2
md"""
#### The Cross-Correlation function

To compute the cross-covariance between two vectors $x_t$ and $y_t$, at the $k$ lag, we do:

$cv_{x y}(k)=\frac{1}{n-1} \sum_{t=1}^{n-k}\left(x_{t+k}-\bar{x}\right)\left(y_{t}-\bar{y}\right) \tag{6}$

where, according to our notation above, $\{\overline{x},\overline{y}\}$ are the sample means of $x_t$ and $y_t$, respectively, while $k$ is a positive lag.

The cross-correlation function is a normalized version of the cross-variance by dividing it by the sample variances of $x_{t}$ and $y_{t}$. As the sample variances also have $n-1$ in their denominators, the terms $n-1$ cancel out and the cross-correlation function can be computed using the formula:

$$cr_{x y}(k)= \frac{\sum_{t=1}^{n-k}\left(x_{t+k}-\bar{x}\right)\left(y_t-\bar{y}\right)} {\sqrt{\sum_{t=1}^n\left(x_t-\bar{x}\right)^2} \times \sqrt{\sum_{t=1}^n\left(y_t-\bar{y}\right)^2}} \tag{7}$$

"""

# ╔═╡ 3c77e1e2-b5f2-4440-b4b1-08149e7d453b


# ╔═╡ 47ab9b42-f7cf-4226-8287-f1efe07f73c8
md"""
#### Julia functions for basic statistical measures
"""

# ╔═╡ 6d85d1f3-e1e4-4a80-89b7-1071b20de7aa
md"""

| Measure                   |   | function                             |    | Package         |    
|:--------------            |:--|:-------                              |:-- |:--------        |
| Mean                      |   | mean(variableX)                      |    | Statistics      |       
| Standard deviation        |   | std(variableX)                       |    | Statistics      |           
| Auto-correlation          |   | autocor(variableX)                   |    | StatsBase       |            
| Cross-correlation         |   | crosscor(variableX,variableY, lags)  |    | StatsBase       |          
| Correlation coefficient   |   | cor(variableX,variableY)             |    | Statistics      |
"""

# ╔═╡ d816ec4a-6478-4508-b7e3-22466a75c5cb
md"""
### Using random observations
"""

# ╔═╡ 95453c1f-1a35-4658-a9fc-60e8bbdfbc3d
md"""
"*In discrete time, white noise is a discrete signal whose samples are regarded as a sequence of serially uncorrelated random observations with zero mean and finite variance; a single realization of white noise is a random shock*". from Wikipedia
"""

# ╔═╡ d4fc54f0-c37a-44f1-84f7-fe21c8e2862a
noise = randn(500);

# ╔═╡ 9fa258be-77bf-4ac2-80fc-a6711977ec28
shrek = randn(500);

# ╔═╡ 990bb081-5591-4b13-9e5a-e93efc3afc76
begin
	fig1 = plot(noise,
		mode = "markers+lines", marker_size = "6", line_color = "BlueViolet", marker_color = "Blue", line_width = 0.5,
		Layout(title_text = "White noise", title_x = 0.5, hovermode = "x", height = 450))
end

# ╔═╡ 59b72098-9005-4f18-a196-a2da0bf50a87
savefig(fig1 , "WhiteNoise.svg") # saving fig1 as a pdf file

# ╔═╡ ace171a3-42b5-4798-a471-873b5a2e1cb7
mean(noise)

# ╔═╡ 4c13cf93-f7d3-4d4c-a4a8-5f7259984a3e
std(noise)

# ╔═╡ c7841346-c3ab-4ca8-9371-b0e6657274e5
autocor(noise)

# ╔═╡ 8bee12a0-7109-4bd0-9681-8b447693984f
begin 
	plot(autocor(noise), 
		mode = "markers+lines", marker_size = "7", line_color = "BlueViolet", marker_color = "Blue", line_width = 0.6,
	Layout(title_text = "Autocorrelation function: white noise", title_x=0.5, height = 450,
	xaxis_title = "lag", yaxis_title = "Autocorrelation")) 
end

# ╔═╡ 5d2d0792-eb31-4e4e-a86a-22d46bd9ee6f
begin
	#using ShiftedArrays for `lag`
	shrek_lagged = ShiftedArrays.lag(shrek, 1) ; # instaed of 1, try different lags 10, 20, 100
	# let's plot 
	plot(shrek, shrek_lagged,
			mode="markers",
			marker_symbol="circle", marker_size="6",line_width= 0.3, 
            marker_color = "Blue",
			Layout(width=600, height=600,
				title_text = "shrek vs shrek_lagged", title_x=0.5,
				xaxis_title = "shrek",
				yaxis_title = "shrek_lagged")
	)
end

# ╔═╡ 2a948df7-6bc0-4ab0-b736-49497b4f3d4c
plot(noise, shrek, 
	mode = "markers", marker_symbol="circle", marker_size="6", marker_color= "Purple",
	Layout(width=600, height=600,
	title_text = "noise vs shrek", title_x=0.5,
	xaxis_title = "noise",
	yaxis_title = "shrek",
	)) 

# ╔═╡ d33ca297-c94a-4ba6-970b-df8bca54d428
clags = -100:100 # to produce the cross correlation plots

# ╔═╡ a9cfeb52-b360-4661-8c59-b6882d04469b
plot(clags, crosscor(noise, shrek, clags), 
			mode="markers+lines",
			marker_symbol="circle", marker_size="6",line_width= 0.3, 
            marker_color = "Blue",
			Layout(title_text = "Cross-correlation between noise and hrek", title_x=0.5, height = 450))

# ╔═╡ f2956bed-9534-4c57-bfaa-fbef3c4af810
md"""
### Using data
"""

# ╔═╡ 65ec39b9-0147-4ab2-a18b-ec330af5cf2f
inflation = CSV.read("ECB Data Inflation.csv", DataFrame)

# ╔═╡ 0d7ff3f1-7e34-4317-857b-c212d4f5989c

begin
	p1 = plot(inflation.Months, inflation.Portugal, 
				mode="markers+lines",
				marker_symbol="circle", marker_size="5",line_width= 0.3,
            	marker_color = "DarkRed")
	relayout!(p1, title_text = "Inflation in Portugal", title_x = 0.5, 
				height = 450, hovermode = "x",
				xaxis_range = ["1996-08-01" , "2024-06-01"])
	p1
end

# ╔═╡ 99cb7dc3-f7d5-4cb5-9301-99ddc860b52c
begin
	trace10 = bar(;x = inflation.Months, y = inflation.EuroArea .-2, height = 450,
				name = "Portugal",
				marker_symbol="circle", marker_size="5",line_width= 0.3,
            	#marker_color = "mediumblue")
				marker_color = "blue")
	
	p100 = plot(trace10)

	
	
	add_shape!(p100, line( x0=Date.("2007-10-01"), y0=-3, x1=Date.("2007-10-01"), y1=8.2,  line=attr(color="RoyalBlue", width=1.5, dash="dashdot")))
	add_shape!(p100, line( x0=Date.("2020-02-01"), y0=-3, x1=Date.("2020-02-01"), y1=8.2,  line=attr(color="RoyalBlue", width=1.5, dash="dashdot")))

	relayout!(p100, height = 450,
		yaxis_title="Deviations from the target inflation rate",
		font_family="Calibri", font_size=15,
		title_text = "Euro Area inflation minus 2% (the target value for inflation)", title_x = 0.5,
		titlefont_size = "22",
		
		annotations=[
			attr(x=Date.("2007-08-01"), y=8.5,
            	text="Financial crisis starts",
            	showarrow=false, font_size=18, 
            	arrowhead=1),
			
        	attr(x=Date.("2020-02-01"), y=8.5, 
				text="Covid19 starts",
            	showarrow=false,
            	yshift=0
        )
    ])
	
	p100
end

# ╔═╡ 1dac18c0-836c-4ee5-a5e4-bd120ce46bc5
md"""
#### Select a subset of a data frame based on years
"""

# ╔═╡ feaab8f3-93c5-4f92-a236-2532a388cd51
begin
	first_date = Date("1997-2-28")
	last_date = Date(1997, 09, 30)
	subset_inflation = inflation[(first_date .<= inflation.Months .<= last_date), :]
end

# ╔═╡ 08d213c0-9be4-48a2-9463-2534a6a5d7ec
md"""
#### Another way of doing it
"""

# ╔═╡ 3ddf9a66-f391-4e30-9168-9dc2e27a49b9
inflation[(@. 1997 <= Dates.year(inflation.Months) <= 1999), :] #@contradict in https://discourse.julialang.org/t/how-to-filter-date-data-column-in-a-dataframe/110485

# ╔═╡ e9a49103-860e-426e-84c7-f41fb3fab9b4
md"""
#### ... still another way of doing it
"""

# ╔═╡ 9a4394a5-928a-4b1a-b40a-3e50e34a92ba
inflation[Date("1997-06-30") .<= inflation.Months .<= Date("1998-06-30"), :] # @Rafael Guerra in https://discourse.julialang.org/t/how-to-filter-date-data-column-in-a-dataframe/110485

# ╔═╡ 121cb8d7-b1e5-454d-8e73-26b4d5fff4ad
md"""
### How to create an artificial data set
"""

# ╔═╡ ab172c4a-5e0b-404b-a522-4f10e1ed4f40
begin
	# Create a sample DataFrame
df = DataFrame(
    Date = collect(Date(1997, 1, 31):Month(1):Date(2023, 12, 31)),
    Value = rand(324))

end;

# ╔═╡ 8f412cf8-bcb6-4cb5-9d59-88e7e5f704d6
df

# ╔═╡ 829e2519-69f1-4ea0-8f68-6a0ba22552b7
begin
	# Define the start and end dates for the subset
	start_date = Date(2002, 7, 31)
	end_date = Date(2005, 6, 30)
	subset_df = df[(start_date .<= df.Date .<= end_date), :]
end

# ╔═╡ a5a1cbe5-cda6-4017-8b7f-3bce54951dc8
describe(inflation)

# ╔═╡ a0541145-d76c-40bb-8a51-17f1d42ead0a
describe(inflation, :mean , :min , :median , :max , :std , :nmissing , :eltype )

# ╔═╡ 495f6688-5a97-4747-9dd2-45009b238cc0
describe(inflation, :all)

# ╔═╡ f1d69f4e-e1f3-490e-9b0a-9d75204fcaac
begin
	trace10_1 = scatter(;x = inflation.Months, y = inflation.EuroArea, 
				name = "EuroZone", mode="markers+lines",
				marker_symbol="circle", marker_size="5",line_width= 0.5, 
            	marker_color = "darkblue")
	
	trace10_2 = scatter(;x = inflation.Months, y = inflation.Portugal, 
				name = "Portugal", mode="markers+lines",
				marker_symbol="circle", marker_size="5",line_width= 0.5,
            	marker_color = "maroon")

	layout10_2 = Layout(;#font_size = 16, 
			height = 450,
					
			title_text = "Inflation in the EuroZone and Portugal", title_x=0.5,
		
			hovermode="x",		
		
            xaxis = attr(
               title = "Monthly obervations",
               tickformat = "%Y",
               hoverformat = "%Y-M%m",
               tick0 = "1997/01/01",
               dtick = "M60" ),
		
        	xaxis_range = [Date.(1997), Date.(2024)],
        	yaxis_title = "Percentage points",
        	#yaxis_range=[-2, 2], 
        	titlefont_size = 16)

	p10_2 = plot([trace10_1, trace10_2], layout10_2)
end

# ╔═╡ 60bab72a-81ac-40a5-b7de-ed822e012f6f
md"""
To obtain the autocorrelation function, we can use the `autocor` function from the `StatsBase.jl` package, as follows:
"""

# ╔═╡ e5c4408d-0174-4f39-8a58-121502c0f193
plot(autocor(inflation.EuroArea), 
			mode="markers+lines",
			marker_symbol="circle", marker_size="7",line_width= 0.3, 
            marker_color = "Blue",
	Layout(title_text = "Auto-correlation of inflation in the EZ", title_x=0.5, hovermode = "x", height = 450)
)

# ╔═╡ db133e97-b597-48f6-a232-180ae9cecbab
md"""
To plot Inflation against lagged-Inflation, we can use a slider to see what happens with different lags. 
"""

# ╔═╡ 6aa9df0f-cf04-4d72-906a-c1c1badf449b
md"""
`lags = `$(@bind lags Slider(-36:1:36, default=0, show_value=true))
"""

# ╔═╡ ffe77616-e884-40b1-8d1e-32b5573c547f
begin
	#using ShiftedArrays
	EZ = inflation.EuroArea
	EZ_lagged = ShiftedArrays.lag(EZ, lags) # try different lags using the slider above: 1, 2, 5 ...
	# let's plot 
	plot(EZ,EZ_lagged,
			mode="markers",
			marker_symbol="circle", marker_size="6",line_width= 0.3, 
            marker_color = "#7a5135",
			Layout(width=600, height=600,
				title_text = "Inflation vs Lagged-Inflation: EZ", title_x=0.5,
				xaxis_title = "EZ inflation",
				yaxis_title = "EZ inflation (lagged)")
	)
end

# ╔═╡ 9b09c201-ffc2-4d10-9958-c9523d60ccae
plot(inflation.EuroArea, inflation.Portugal, 
	mode = "markers", marker_symbol="circle", marker_size="6",marker_color= "blue",
	Layout(width=600, height=600,
		title_text = "Inflation in the EuroZone and Portugal", 
		title_x=0.5,
		xaxis_title = "EuroZone",
		yaxis_title = "Portugal",
		yaxis_range=[-1.0 , 5.5],
		xaxis_range=[-1.0 , 6.5]
	)) 

# ╔═╡ 5d515df6-6487-4a0c-9e05-8df53631c0d5
cor(inflation.EuroArea, inflation.Portugal)

# ╔═╡ ec20be10-13c3-45a0-ba9c-5b986740f658
plot(clags, crosscor(inflation.EuroArea, inflation.Portugal, clags), 
			mode="markers+lines",
			marker_symbol="circle", marker_size="6",line_width= 0.3, 
            marker_color = "Blue",
			Layout(title_text = "Cross-correlation between inflation in the EuroZone and Portugal", title_x=0.5, height = 450))

# ╔═╡ e0b538a5-5cd8-4a89-a8de-b6e9f339add9
ccf_julia=crosscor(inflation.EuroArea, inflation.Portugal, clags)

# ╔═╡ 7b3ec0c9-0f2e-406e-a76d-bbea872c2b65
CSV.write("ccf_julia.csv", (ccf_julia=ccf_julia,))

# ╔═╡ 650e8dc7-31b7-4a97-b4fa-24e27c9da1ca
typeof(ccf_julia)

# ╔═╡ 24b95a91-d2f8-435f-80a8-764f1534c3eb
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 52c76dc2-146c-4178-9aa3-d132a0551168
md"""
## 2. Stability of dynamic systems
"""

# ╔═╡ fff5ce25-8546-43b5-9d82-ec3c560105dd
md"""

A dynamical process displays the evolution of one variable (or a set of variables) across time, starting from a particular initial value (or initial values), resulting from the impact of a set of given forces.

It has three fundamental ingredientes:

- An intial condition

- Tansitional dynamics

- A steady state.

See the following figure for a graphical example.

"""

# ╔═╡ b48ad882-52b8-4809-93ae-ad22b5705e3c
Resource("https://vivaldomendes.org/images/depot/transitional_dynamics.png", :width=>650)

# ╔═╡ c3ea937d-f8db-4e3d-b12d-50957efb3040
md"""
Consider the simplest possible dynamic process: an Auto_Regressive process of first order, usually called simply by an AR(1):

$$y_{t+1}= a + ρ \cdot y_t \tag{8}$$ 

where $a$ is a constant (or an exogenous force) and $\rho$ is a parameter.
"""

# ╔═╡ 652c8219-1530-47b1-adcb-17696171bab6
md"""
👉 a) Pass the model's data $(a,\rho)$ into the notebook.
"""

# ╔═╡ 75ca182d-df13-4248-9d4c-5f572fc89637
begin
	a = 10
	ρ = 0.5
end

# ╔═╡ bf61c3c3-bba6-4417-8a11-3603c6c20378
md"""
👉 **b)** Calculate the steady state of this model. Remember that the steady state is obtained by imposing the condition $y_{t+1} = y_t = \bar{y}$ upon our AR(1). Therefore, 

$$\bar{y}=a+\rho \bar{y} \Rightarrow \bar{y}=\frac{1}{1-\rho}a=(1-\rho)^{-1} \cdot a \tag{9}$$
"""

# ╔═╡ 8fcbd455-bd06-4fe7-96cb-a15de9e80d79
ȳ = inv(1-ρ) * a

# ╔═╡ 4847b31b-1d00-44d8-b1da-745559807a9a
md"""
!!! tip "Answer (b)"

	Here

"""

# ╔═╡ dafa708f-e667-4c1b-ac44-5c639895f60c
md"""
👉 **c)** Check the stability of this model.
"""

# ╔═╡ af7929eb-ab80-4275-bd45-09c95dc22079
eigvals(1-ρ)

# ╔═╡ e63efa59-a6f2-41c0-b5a1-a82630636f65
md"""
!!! tip "Answer (c)"

	Here

"""

# ╔═╡ bfb7998b-4445-4023-af98-ae0790af0b8c
md"""
👉 **d)** Simulate the model for a period of 30 iteration, with an intitial condition of $y_1 = 4$.
"""

# ╔═╡ 00b75cd1-8a92-4523-9ff2-d180bf25e869
begin
	n = 30									# number of iterations we want to simulate
	y1 = 4 									# initial condition
	l = length(y1)							# number of initial conditions
	y = [y1  zeros(l, n-1)] 				# Allocating space for y

	for t = 1 : n-1 						# begins the "for" loop
		y[t+1] = a + ρ * y[t] 				# the loop works trogh eq. (8): y(t+1) = a + ρy(t)
	end										# ends the for loop
end

# ╔═╡ 3d2c9595-3cde-4279-a7ee-2efa608998f6
plot(y' , mode = "lines+markers", marker_symbol = "circle", marker_size = 7, marker_color = "Blue", line_width = 0.5,
	Layout(
	title_text = "The evolution of an AR(1) process", title_x=0.5,
	xaxis_title = "time",
	yaxis_title = "y(t)",
	height = 450,
	)) 

# ╔═╡ 268cab98-86dc-4070-a285-d00950a089b1


# ╔═╡ cee5cafb-07a7-4804-8589-ddae72a439a8
md"""
### Exercise 1: add noise
"""

# ╔═╡ 7f962bfb-1773-4179-882e-d17686b4ea50
md"""
Introduce a series of stochastic (or random) shocks to the AR(1) process above. This stochastic variable is defined below as $\varepsilon$ and the process is written as:

$$y_{t+1}= a + ρ \cdot y_t + \varepsilon_{t+1} \tag{10}$$ 

The impact of this random element can be obtained by adding the term `ε[t+1]` to the loop above. Now, the loop should be written as:

```julia
		y[t+1] = a + ρ * y[t] + ε[t+1]
```
"""

# ╔═╡ cacc9719-25a9-4b03-b92a-07bbca9e0acd
md"""
To avoid multiple definitions in the notebook, let us name our variable "yₙ" instead of "y," where "yₙ" represents "y with noise". Everything else will remain the same, apart from telling the notebook the kind of noise we want in the model. In this case, we will use n random normally distributed observations (randn(n)).
"""

# ╔═╡ 93979e82-668f-416b-bec4-f6cea410a83d
begin 										# this block uses special characters of class: \bi 
	𝒏 = 30									# number of iterations we want to simulate
	𝒚1 = 4 									# initial condition
	𝒍 = length(𝒚1)							# number of initial conditions
	𝜺 = randn(𝒏)  							# Random shocks
	𝒚 = [𝒚1  zeros(𝒍, 𝒏-1)] 				# Allocating space for y

	for t = 1 : n-1 						# begins the "for" loop
		𝒚[t+1] = a + ρ * 𝒚[t] + 𝜺[t+1]		# the loop works trogh eq. (8): y(t+1) = a + ρy(t)
	end										# ends the for loop
end

# ╔═╡ 82c4d622-febe-49ca-b827-5ae443f3d561
md"""
Now, we can plot our variable $y$ with noise:
"""

# ╔═╡ 9b07963e-a523-490d-aedd-c9d59b6dff84
plot(𝒚',
	mode = "lines+markers", marker_symbol = "circle", marker_size = "7", marker_color = "Blue", line_width = 0.5,
	Layout(height = 450,
	title_text = "The evolution of an AR(1) process with noise", title_x=0.5,
	xaxis_title = "time",
	yaxis_title = "y(t)",
	)) 

# ╔═╡ 625cf0da-c085-4d1e-98ff-5e5091589923
md"""
### Exercise 2: public debt
"""

# ╔═╡ 40de1d39-effa-460b-8ccd-85d5be501a63
md"""

The sustainability of public debt is given by this simple linear difference equation:

$$d_{t+1}=p+\left(\frac{1+r}{1+g}\right) d_{t} \tag{11}$$

where

-  $d_t$ is the public debt as a percentage of GDP
-  $p$ is the primary deficit as a percentage of GDP 
-  $r$ is the real interest rate paid on public debt
-  $g$ is the growth rate of real GDP

Let us simulate the dynamics associated with this equation using a `for` loop. By changing the sliders below, present a scenario that is capable of reducing the level of $d_t$ to 80% of GDP in no more than fifty years.

"""

# ╔═╡ e3772aa9-c1f1-4989-b0f6-50c12a6fe89c
md"""
### Exercise 3: add noise to the primary deficit
"""

# ╔═╡ 09b9db62-c44e-4bdb-964f-4c95c73d417a
md"""
To add noise to the dynamic process that describes the sustainability of public debt, add the term $\epsilon_{t+1}$ to eq. (3), where $\epsilon$ is meant to represent a sequence of random factors that affect the level of public debt over time:

$$d_{t+1}=p+\left(\frac{1+r}{1+g}\right) d_{t} + \epsilon_{t+1} \tag{12}$$

The loop will be written as:

```julia
	d[t+1] = p + ((1+r)/(1+g)) * d[t] + 𝝐[t+1]
```
"""

# ╔═╡ e6b89599-8a85-455e-a96b-9d3ddeeb3000
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 13cb0884-742b-4444-a84f-1290ebf2b90e
md"""
## 3. Higher dimension dynamic models
"""

# ╔═╡ b641186e-e584-4f21-967d-48bc39f25569
md"""
The stability of a dynamic system depends on the **eigenvalues** of the characteristic matrix of the system.
In our simple AR(1) model

$$y_{t+1}=a+\rho ⋅ y_{t}$$



there is only one eigenvalue, which corresponds to parameter $\rho$. As the example above shows, if $|\rho|<1$ the process has a unique and stable equilibrium. On the other hand, if  $|\rho|>1$ the process will have an equilibrium, but the former will be unstable. Finally, if $|\rho|=1$ there will be no equilibrium in this process.

A similar reasoning can be applied to our rather more general model:

$$Ax_{t+1}= Z + Bx_{t}+ C\varepsilon_{t} \tag{13}$$

where $A, B , C$ are $n\times n$ matrices, and $x_{t+1}, x_{t},\varepsilon_{t}, Z$ are $n\times 1$ vectors. Notice also that in order to simulate the model we have to apply:

$A^{-1}B=D \ , \ A^{-1}C=E \ , \ A^{-1}Z * H.$

The point is that now we will **_not have_** just one eigenvalue. The number of eigenvalues will be dependent on the number of variables in the system (or the number of equations). For example, if our system has 3 variables (and consequently, 3 equations) matrix $D$ will have a dimension of 3x3 and there will be three eigenvalues ($\lambda_i, i=1,2,3.$). If all $|\lambda_i| <1$, the system will converge to a stable equilibrium, if (at least) one of them in modulus is larger than $1$ the system has an unstable equilibrium, and no equilibrium if one of them is equal to $1$ in modulus.

See next figure for a summary of the stability conditions of a dynamic process of **dimension 2** (so we have two eigenvalues). For the case where we have three eigenvalues, the reasoning is similar.
	"""

# ╔═╡ 585f0c52-1344-4d60-89fd-a59bbbf40ed9
Resource("https://ebs.de.iscte.pt/stability.png", :width=>600)

# ╔═╡ 9aef5e9f-bfcb-4489-8eef-954ad4dfdc0a


# ╔═╡ 1e532543-f979-4d8a-81db-a2cfaf4018e9
md"""
#### A Vector Auto-Regressive model of order 3: a VAR(3) model

"""

# ╔═╡ 800793c6-201a-4d09-b704-859fb90cc138
md"""
Consider the following model 

$x_{t+1}=D x_{t}+E \varepsilon_{t} \tag{14}$

where 

$$x_{t+1} =
  \left[ {\begin{array}{c}
   z_{t+1}\\
   y_{t+1}\\
   v_{t+1}
   \end{array} } \right] \tag{15}$$
and matrices $D$ and $E$ are given by:


$$D =  \left[ {\begin{array}{ccc}
   0.97 & 0.10 & -0.05 \\
   -0.3 & 0.98 & 0.05 \\
   0.01 &-0.04 & 0.96
  \end{array} } \right] \qquad , \qquad

E =
  \left[ {\begin{array}{ccc}
   0 & 0 & 0 \\
   0 & 0 & 0 \\
   0 &0 & 0
  \end{array} } \right] \tag{16}$$


Finally, please assume that the initial state of our system (or its initial conditions) are: $z_1=1$, $y_1=0$, and $v_1=-1$. For clarity, we can write these conditions using a row vector: $$x_1 = [1,0,-1]$$.

"""

# ╔═╡ dd1a41d5-de1b-4a0f-82fa-96ce46972145
md"""
👉 Let us simulate our system for 200 iterations to glimpse its dynamics. Firstly, we have to pass the entries of matrix $D$ into the notebook. Secondly, we have to write the `for loop,` and finally, we will get the plot.
"""

# ╔═╡ 2cdee962-2074-4f04-905a-6c2109747d4f
md"""
Matrix $D$:
"""

# ╔═╡ bf81dbd0-6181-4cd2-aba1-31a72c4bce0b
D = [ 0.97   0.10  -0.05 ; -0.3  0.98  0.05 ; 0.01  -0.04  0.96 ]

# ╔═╡ 7b542867-7e3d-430e-afe4-c70726fb833c
md"""
The foor loop:
"""

# ╔═╡ 2f65205b-5410-435a-a771-6c5d746b2472
begin 										# this block uses special characters of class: \bb
	ℕ = 200									# Number of iterations
	𝕩1 = [1, 0, -1] 						# Set of initial conditions
	𝕃 = length(𝕩1) 							# Number of initial conditions
	𝕩 = [𝕩1  zeros(𝕃, ℕ-1)]; 				# Alocating space for x
	
	for t = 1 : ℕ-1  						# begins the "for" loop
	   𝕩[:, t+1] = D * 𝕩[:, t]         		# works trough eq. (14); no E because all its elements are 0.
	end 									# ends the "for" loop
end	

# ╔═╡ 55962895-c38a-474b-b737-42512b891986
md"""
The plot:
"""

# ╔═╡ f5aa880c-06f8-4a26-83ac-097db126eabc
begin
	p_1 = plot(𝕩', mode = "lines",
	Layout(hovermode = "x", title_text = "A VAR(3) Model", title_x=0.5, height =450,
		xaxis_title = "time",
		yaxis_title = "z(t) , y(t) , v(t)"	)) 
	restyle!(p_1, 1:3, name=["z(t)" , "y(t)" , "v(t)"], line_color = ["RoyalBlue", "maroon", "red"])
	p_1
end

# ╔═╡ f3e3c449-1642-419d-9fd3-d7f7297f812f
md"""
👉 Checking stability: eigenvalues.
"""

# ╔═╡ 66936f30-7c10-47b2-8ab3-47c8d8a40e59
eigvals(D)

# ╔═╡ da40f057-fc5c-4f82-be14-1d1cf9a00204
md"""
Checking whether the condition $\sqrt(\alpha^2+\beta^2)$ is greater than 1, less than 1, or equal to 1:
"""

# ╔═╡ 12586f44-f402-4c28-ba7d-00b418fbeaf0
Cond_D = sqrt(0.982782^2 + 0.181356^2)

# ╔═╡ 84038536-234c-46e9-ae0c-cc755d92a9d9
md"""
An easier way is to use the "abs" function in Julia. It immediately gives the results we are looking for:
"""

# ╔═╡ 3cbefd1b-8892-43a1-affb-117613d2dadf
abs.(eigvals(D))

# ╔═╡ 330f9526-66f9-4cfc-a3f0-61d58af7b9b5
md"""
Therefore, as $|\sqrt{0.944435^2-0^2}|<1$, and $|0.999375|<1$, this VAR(3) model is stable. However, its kind of stability is a very special one: it takes a huge amount of time to converge to its steady-state.
"""

# ╔═╡ 66779d90-b2ab-4691-9e2e-b7333729e708
md"""
👉 Finally, simulate the same model with one single change: instead of having D[1,1]=0.97, change it to D[1,1]=0.99. Check the eigenvalues with this new parameter value.
"""

# ╔═╡ ad23ba06-2cb7-4ec9-a1c5-32f78b2ee145
md"""
### Exercise 4: a model of dimension 2
"""

# ╔═╡ be339830-24b1-4011-b626-589b0158399d
md"""
Let's see what happens in the following model with dimension 2, and with a vector including constants $(Z)$:

$Y_{t+1}= Z+ B \cdot Y_{t} \tag{17}$

$B=\left[ {\begin{array}{cc}
   1.1 & -0.4  \\
   0.5 & 0.2  \\
   \end{array} } \right] \tag{18}$

$Z = \left[ {\begin{array}{c}
   10  \\
   -25 \\
   \end{array} } \right] \tag{19}$
"""

# ╔═╡ 659da403-d7d0-47c7-bea1-237c4b09e93d
md"""
👉 a) Pass the entries of the matrices $C,Z$ into the notebook.
"""

# ╔═╡ 58d0be12-843e-46cb-8c74-c8c3de4a45db
B = [1.1  -0.4 ; 0.5   0.2]

# ╔═╡ 8f1471b1-0471-4890-9469-41f1369bc0ff
Z = [10 ; -25]

# ╔═╡ eb76ac98-8640-4eea-a616-3c867229fe60
md"""
👉 **b)** Calculate the steady state of this model.
"""

# ╔═╡ 2215a118-0a8d-454a-9f48-bd23208c0d81
Ȳ = inv(I-B)*Z

# ╔═╡ 9b4dd09d-b1ea-4041-a097-24e8e10737ad
md"""
!!! tip "Answer (b)"

	Here

"""

# ╔═╡ 8e7e6b3a-a2be-4e20-b88c-1f4dfe00be43
md"""
👉 **c)** Check the stability ofthis model.
"""

# ╔═╡ 9871cd1b-bbba-40d9-96f5-c388d20cc4c6
eigvals(B)

# ╔═╡ 2f8087f3-0bbe-4715-9cf3-feb02ab3da95
md"""
!!! tip "Answer (c)"

	Here

"""

# ╔═╡ 138cf359-5a5e-4417-a6d9-d2b86f6d5012
md"""
👉 **d)** Simulate the model for a period of 30 iterations.
"""

# ╔═╡ 539af3d5-9430-47bc-8df0-fd8d5f575b59
begin 													# this block uses normal characters
	iterations = 30 									# Number of iterations
	Y1 = [50 ; 160] 									# Setting the initial coniditions
	m = length(Y1) 										# Number of initial conditions
	Y = [Y1  zeros(m, iterations-1)]; 					# Alocating space for Y
	
		for t = 1 : iterations-1  						# begins the "for" loop
	    	Y[:, t+1] = Z + B * Y[:, t] 				# works trough eq. (17)
		end 											# ends the "for" loop
end

# ╔═╡ bd8d669a-ee76-447d-9739-6428251c92ee
begin	
	fig10 = plot(Y', 
		mode = "markers+lines", marker_size= 6 , line_width= 0.3,
		Layout(hovermode = "x", title_text = "A 2-Dimensional System", title_x=0.5, height = 450,
		xaxis_title = "time",
		yaxis_title = "y1(t) , y2(t)"))
	restyle!(fig10, 1:2, name=["y1(t)" , "y2(t)"], line_color = ["Blue", "red"], marker_symbol= ["circle", "circle"])
	fig10
end

# ╔═╡ edbd2b1c-1452-41a6-962c-3434bd420020
md"""
## Auxiliary cells (please do not change them)
"""

# ╔═╡ d9672a6f-97af-43d1-9a30-50495204fec8
 TableOfContents()

# ╔═╡ f6283c43-c912-430d-a63d-bba0df690cd4
md"### CSS code"

# ╔═╡ c59c18ff-f4a4-4b28-8e5a-2c7a03e86d83
begin
      struct TwoColumns{L, R}
    left::L
    right::R
      end
      
      function Base.show(io, mime::MIME"text/html", tc::TwoColumns)
          write(io, """<div style="display: flex;"><div style="flex: 48%;margin-right:2%;">""")
          show(io, mime, tc.left)
          write(io, """</div><div style="flex: 48%;">""")
          show(io, mime, tc.right)
          write(io, """</div></div>""")
      end
end

# ╔═╡ 5ec7390c-2383-4033-9089-1c051b806912
TwoColumns(
md"""

`𝐩 = `$(@bind 𝐩 Slider(-5.0:0.1:5.0, default=0.5, show_value=true)) ... Primary deficit

`𝐫 = `$(@bind 𝐫 Slider(0.0:0.05:4.0, default=1.5, show_value=true))  ... Real interest rate

""",
	
md"""

`𝐠 = `$(@bind 𝐠 Slider(0.0:0.05:5.0, default=2.0, show_value=true)) ... GDP growth rate

`𝐝1 = `$(@bind 𝐝1 Slider(0.0:0.1:2, default=1.2, show_value=true)) ... Initial value

"""
)

# ╔═╡ 8892c48d-5647-4ecf-9edb-51370332d171
begin 																# this block uses special characters of class: \bf 
	𝐍 = 500 														# Number of iterations 
	# 𝐝1 															# the initial condition is set by the 𝐝1-slider below
	𝐋 = length(𝐝1) 													# Number of initial conditions 
	𝐝 = [𝐝1  zeros(𝐋, 𝐍-1)] 										# Allocating space for y

	for t = 1 : 𝐍-1 												# begins the "for" loop 
		𝐝[t+1] = 𝐩/100 + ((1 + 𝐫/100)/(1 + 𝐠/100)) * 𝐝[t]     		# Eq. (11) where p,r,g are expressed in decimal units
	end 															# ends the "for" loop
end
 

# ╔═╡ 095691a9-2b67-439e-a19f-61312496b3b5
begin
	layout10_91 = Layout(;
		height = 450,
		
		title_text = "The evolution of Public Debt as % of GDP", 
		title_x =0.5,
		
		hovermode="x",
		
		
        xaxis = attr(
               title = "Years", 
               #tickformat = "%Y",
               #hoverformat = "%Y-M%m",
               #tick0 = "1997/10/01",
               ),
		
        #xaxis_range = [2000, 2020],
        yaxis_title = "Debt / GDP",
        #yaxis_range=[-2, 2], 
        titlefont_size = 16)
	plot(𝐝', mode = "lines", line_width = "2.5", layout10_91)
end

# ╔═╡ 6857086e-c1d2-49e4-b54d-2b8267b1664f
begin    																# this block uses special characters of class: \it
	𝑁 = 500 															# number of iterations 
	𝐿 = length(𝐝1) 														# The number of initial conditions
	𝜖 = randn(𝑁) 														# Random shocks 
	𝑑 = [𝐝1  zeros(𝐿, 𝑁-1)] 											# Allocating space for d_noise

	for t = 1 : 𝑁-1 													# begins the "for" loop 
		𝑑[t+1] = 𝐩/100 + ((1+ 𝐫/100)/(1+ 𝐠/100))* 𝑑[t] + 𝜖[t+1]/100		# works through eq. (12) expressed as decimal units
	end 																# ends the "for" loop
end

# ╔═╡ e77e17bb-4275-4e59-9478-79231bac359c
begin
	fig2 = Plot(𝑑')
	relayout!(fig2, height = 450,
		Layout(title_text = "The evolution of Public Debt as % of GDP with Noise", 
		title_x =0.5,))
	fig2
end

# ╔═╡ 021ee205-8123-4158-9c75-f1cc7f2ab6f2
html"""
<style>
	@media screen {
		main {
			margin: 0 auto;
			max-width: 1600px;
    		padding-left: max(100px, 10%);
    		padding-right: max(383px, 10%); 
            # 383px to accomodate TableOfContents(aside=true)
		}
	}
</style>
"""

# ╔═╡ 00000000-0000-0000-0000-000000000001
PLUTO_PROJECT_TOML_CONTENTS = """
[deps]
CSV = "336ed68f-0bac-5ca0-87d4-7b16caf5d00b"
DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
Dates = "ade2ca70-3891-5945-98fb-dc099432e06a"
LaTeXStrings = "b964fa9f-0449-5b57-a5c2-d3ea65f4040f"
Latexify = "23fbe1c1-3f47-55db-b15f-69d7ec21a316"
LinearAlgebra = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"
PlotlyKaleido = "f2990250-8cf9-495f-b13a-cce12b45703c"
PlutoPlotly = "8e989ff0-3d88-8e9f-f020-2b208a939ff0"
PlutoUI = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
ShiftedArrays = "1277b4bf-5013-50f5-be3d-901d8477a67a"
Statistics = "10745b16-79ce-11e8-11f9-7d13ad32a3b2"
StatsBase = "2913bbd2-ae8a-5f71-8c99-4fb6c76f3a91"

[compat]
CSV = "~0.10.15"
DataFrames = "~1.8.1"
LaTeXStrings = "~1.4.0"
Latexify = "~0.16.10"
PlotlyKaleido = "~2.3.1"
PlutoPlotly = "~0.6.5"
PlutoUI = "~0.7.79"
ShiftedArrays = "~2.0.0"
StatsBase = "~0.34.10"
"""

# ╔═╡ 00000000-0000-0000-0000-000000000002
PLUTO_MANIFEST_TOML_CONTENTS = """
# This file is machine-generated - editing it directly is not advised

julia_version = "1.10.10"
manifest_format = "2.0"
project_hash = "c589a07c6e54acc4bd68d4951fd891ce07e70deb"

[[deps.AbstractPlutoDingetjes]]
deps = ["Pkg"]
git-tree-sha1 = "6e1d2a35f2f90a4bc7c2ed98079b2ba09c35b83a"
uuid = "6e696c72-6542-2067-7265-42206c756150"
version = "1.3.2"

[[deps.AliasTables]]
deps = ["PtrArrays", "Random"]
git-tree-sha1 = "9876e1e164b144ca45e9e3198d0b689cadfed9ff"
uuid = "66dad0bd-aa9a-41b7-9441-69ab47430ed8"
version = "1.1.3"

[[deps.ArgTools]]
uuid = "0dad84c5-d112-42e6-8d28-ef12dabb789f"
version = "1.1.1"

[[deps.Artifacts]]
uuid = "56f22d72-fd6d-98f1-02f0-08ddc0907c33"

[[deps.Base64]]
uuid = "2a0f44e3-6c83-55bd-87e4-b1978d98bd5f"

[[deps.CSV]]
deps = ["CodecZlib", "Dates", "FilePathsBase", "InlineStrings", "Mmap", "Parsers", "PooledArrays", "PrecompileTools", "SentinelArrays", "Tables", "Unicode", "WeakRefStrings", "WorkerUtilities"]
git-tree-sha1 = "deddd8725e5e1cc49ee205a1964256043720a6c3"
uuid = "336ed68f-0bac-5ca0-87d4-7b16caf5d00b"
version = "0.10.15"

[[deps.CodecZlib]]
deps = ["TranscodingStreams", "Zlib_jll"]
git-tree-sha1 = "962834c22b66e32aa10f7611c08c8ca4e20749a9"
uuid = "944b1d66-785c-5afd-91f1-9de20f533193"
version = "0.7.8"

[[deps.ColorSchemes]]
deps = ["ColorTypes", "ColorVectorSpace", "Colors", "FixedPointNumbers", "PrecompileTools", "Random"]
git-tree-sha1 = "b0fd3f56fa442f81e0a47815c92245acfaaa4e34"
uuid = "35d6a980-a343-548e-a6ea-1d62b119f2f4"
version = "3.31.0"

[[deps.ColorTypes]]
deps = ["FixedPointNumbers", "Random"]
git-tree-sha1 = "67e11ee83a43eb71ddc950302c53bf33f0690dfe"
uuid = "3da002f7-5984-5a60-b8a6-cbb66c0b333f"
version = "0.12.1"

    [deps.ColorTypes.extensions]
    StyledStringsExt = "StyledStrings"

    [deps.ColorTypes.weakdeps]
    StyledStrings = "f489334b-da3d-4c2e-b8f0-e476e12c162b"

[[deps.ColorVectorSpace]]
deps = ["ColorTypes", "FixedPointNumbers", "LinearAlgebra", "Requires", "Statistics", "TensorCore"]
git-tree-sha1 = "8b3b6f87ce8f65a2b4f857528fd8d70086cd72b1"
uuid = "c3611d14-8923-5661-9e6a-0046d554d3a4"
version = "0.11.0"

    [deps.ColorVectorSpace.extensions]
    SpecialFunctionsExt = "SpecialFunctions"

    [deps.ColorVectorSpace.weakdeps]
    SpecialFunctions = "276daf66-3868-5448-9aa4-cd146d93841b"

[[deps.Colors]]
deps = ["ColorTypes", "FixedPointNumbers", "Reexport"]
git-tree-sha1 = "37ea44092930b1811e666c3bc38065d7d87fcc74"
uuid = "5ae59095-9a9b-59fe-a467-6f913c188581"
version = "0.13.1"

[[deps.Compat]]
deps = ["TOML", "UUIDs"]
git-tree-sha1 = "9d8a54ce4b17aa5bdce0ea5c34bc5e7c340d16ad"
uuid = "34da2185-b29b-5c13-b0c7-acf172513d20"
version = "4.18.1"
weakdeps = ["Dates", "LinearAlgebra"]

    [deps.Compat.extensions]
    CompatLinearAlgebraExt = "LinearAlgebra"

[[deps.CompilerSupportLibraries_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "e66e0078-7015-5450-92f7-15fbd957f2ae"
version = "1.1.1+0"

[[deps.Crayons]]
git-tree-sha1 = "249fe38abf76d48563e2f4556bebd215aa317e15"
uuid = "a8cc5b0e-0ffa-5ad4-8c14-923d3ee1735f"
version = "4.1.1"

[[deps.DataAPI]]
git-tree-sha1 = "abe83f3a2f1b857aac70ef8b269080af17764bbe"
uuid = "9a962f9c-6df0-11e9-0e5d-c546b8b5ee8a"
version = "1.16.0"

[[deps.DataFrames]]
deps = ["Compat", "DataAPI", "DataStructures", "Future", "InlineStrings", "InvertedIndices", "IteratorInterfaceExtensions", "LinearAlgebra", "Markdown", "Missings", "PooledArrays", "PrecompileTools", "PrettyTables", "Printf", "Random", "Reexport", "SentinelArrays", "SortingAlgorithms", "Statistics", "TableTraits", "Tables", "Unicode"]
git-tree-sha1 = "d8928e9169ff76c6281f39a659f9bca3a573f24c"
uuid = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
version = "1.8.1"

[[deps.DataStructures]]
deps = ["OrderedCollections"]
git-tree-sha1 = "e357641bb3e0638d353c4b29ea0e40ea644066a6"
uuid = "864edb3b-99cc-5e75-8d2d-829cb0a9cfe8"
version = "0.19.3"

[[deps.DataValueInterfaces]]
git-tree-sha1 = "bfc1187b79289637fa0ef6d4436ebdfe6905cbd6"
uuid = "e2d170a0-9d28-54be-80f0-106bbe20a464"
version = "1.0.0"

[[deps.Dates]]
deps = ["Printf"]
uuid = "ade2ca70-3891-5945-98fb-dc099432e06a"

[[deps.DelimitedFiles]]
deps = ["Mmap"]
git-tree-sha1 = "9e2f36d3c96a820c678f2f1f1782582fcf685bae"
uuid = "8bb1440f-4735-579b-a4ab-409b98df4dab"
version = "1.9.1"

[[deps.DocStringExtensions]]
git-tree-sha1 = "7442a5dfe1ebb773c29cc2962a8980f47221d76c"
uuid = "ffbed154-4ef7-542d-bbb7-c09d3a79fcae"
version = "0.9.5"

[[deps.Downloads]]
deps = ["ArgTools", "FileWatching", "LibCURL", "NetworkOptions"]
uuid = "f43a241f-c20a-4ad4-852c-f6b1247861c6"
version = "1.6.0"

[[deps.FilePathsBase]]
deps = ["Compat", "Dates"]
git-tree-sha1 = "3bab2c5aa25e7840a4b065805c0cdfc01f3068d2"
uuid = "48062228-2e41-5def-b9a4-89aafe57970f"
version = "0.9.24"
weakdeps = ["Mmap", "Test"]

    [deps.FilePathsBase.extensions]
    FilePathsBaseMmapExt = "Mmap"
    FilePathsBaseTestExt = "Test"

[[deps.FileWatching]]
uuid = "7b1f6079-737a-58dc-b8bc-7a2ca5c1b5ee"

[[deps.FixedPointNumbers]]
deps = ["Statistics"]
git-tree-sha1 = "05882d6995ae5c12bb5f36dd2ed3f61c98cbb172"
uuid = "53c48c17-4a7d-5ca2-90c5-79b7896eea93"
version = "0.8.5"

[[deps.Format]]
git-tree-sha1 = "9c68794ef81b08086aeb32eeaf33531668d5f5fc"
uuid = "1fa38f19-a742-5d3f-a2b9-30dd87b9d5f8"
version = "1.3.7"

[[deps.Future]]
deps = ["Random"]
uuid = "9fa8497b-333b-5362-9e8d-4d0656e87820"

[[deps.Ghostscript_jll]]
deps = ["Artifacts", "JLLWrappers", "JpegTurbo_jll", "Libdl", "Zlib_jll"]
git-tree-sha1 = "38044a04637976140074d0b0621c1edf0eb531fd"
uuid = "61579ee1-b43e-5ca0-a5da-69d92c66a64b"
version = "9.55.1+0"

[[deps.HashArrayMappedTries]]
git-tree-sha1 = "2eaa69a7cab70a52b9687c8bf950a5a93ec895ae"
uuid = "076d061b-32b6-4027-95e0-9a2c6f6d7e74"
version = "0.2.0"

[[deps.Hyperscript]]
deps = ["Test"]
git-tree-sha1 = "179267cfa5e712760cd43dcae385d7ea90cc25a4"
uuid = "47d2ed2b-36de-50cf-bf87-49c2cf4b8b91"
version = "0.0.5"

[[deps.HypertextLiteral]]
deps = ["Tricks"]
git-tree-sha1 = "7134810b1afce04bbc1045ca1985fbe81ce17653"
uuid = "ac1192a8-f4b3-4bfe-ba22-af5b92cd3ab2"
version = "0.9.5"

[[deps.IOCapture]]
deps = ["Logging", "Random"]
git-tree-sha1 = "0ee181ec08df7d7c911901ea38baf16f755114dc"
uuid = "b5f81e59-6552-4d32-b1f0-c071b021bf89"
version = "1.0.0"

[[deps.InlineStrings]]
git-tree-sha1 = "8f3d257792a522b4601c24a577954b0a8cd7334d"
uuid = "842dd82b-1e85-43dc-bf29-5d0ee9dffc48"
version = "1.4.5"

    [deps.InlineStrings.extensions]
    ArrowTypesExt = "ArrowTypes"
    ParsersExt = "Parsers"

    [deps.InlineStrings.weakdeps]
    ArrowTypes = "31f734f8-188a-4ce0-8406-c8a06bd891cd"
    Parsers = "69de0a69-1ddd-5017-9359-2bf0b02dc9f0"

[[deps.InteractiveUtils]]
deps = ["Markdown"]
uuid = "b77e0a4c-d291-57a0-90e8-8db25a27a240"

[[deps.InvertedIndices]]
git-tree-sha1 = "6da3c4316095de0f5ee2ebd875df8721e7e0bdbe"
uuid = "41ab1584-1d38-5bbf-9106-f11c6c58b48f"
version = "1.3.1"

[[deps.IrrationalConstants]]
git-tree-sha1 = "b2d91fe939cae05960e760110b328288867b5758"
uuid = "92d709cd-6900-40b7-9082-c6be49f344b6"
version = "0.2.6"

[[deps.IteratorInterfaceExtensions]]
git-tree-sha1 = "a3f24677c21f5bbe9d2a714f95dcd58337fb2856"
uuid = "82899510-4779-5014-852e-03e436cf321d"
version = "1.0.0"

[[deps.JLLWrappers]]
deps = ["Artifacts", "Preferences"]
git-tree-sha1 = "0533e564aae234aff59ab625543145446d8b6ec2"
uuid = "692b3bcd-3c85-4b1f-b108-f13ce0eb3210"
version = "1.7.1"

[[deps.JSON]]
deps = ["Dates", "Logging", "Parsers", "PrecompileTools", "StructUtils", "UUIDs", "Unicode"]
git-tree-sha1 = "b3ad4a0255688dcb895a52fafbaae3023b588a90"
uuid = "682c06a0-de6a-54ab-a142-c8b1cf79cde6"
version = "1.4.0"

    [deps.JSON.extensions]
    JSONArrowExt = ["ArrowTypes"]

    [deps.JSON.weakdeps]
    ArrowTypes = "31f734f8-188a-4ce0-8406-c8a06bd891cd"

[[deps.JpegTurbo_jll]]
deps = ["Artifacts", "JLLWrappers", "Libdl"]
git-tree-sha1 = "b6893345fd6658c8e475d40155789f4860ac3b21"
uuid = "aacddb02-875f-59d6-b918-886e6ef4fbf8"
version = "3.1.4+0"

[[deps.Kaleido_jll]]
deps = ["Artifacts", "JLLWrappers", "Libdl", "Pkg"]
git-tree-sha1 = "43032da5832754f58d14a91ffbe86d5f176acda9"
uuid = "f7e6163d-2fa5-5f23-b69c-1db539e41963"
version = "0.2.1+0"

[[deps.LaTeXStrings]]
git-tree-sha1 = "dda21b8cbd6a6c40d9d02a73230f9d70fed6918c"
uuid = "b964fa9f-0449-5b57-a5c2-d3ea65f4040f"
version = "1.4.0"

[[deps.Latexify]]
deps = ["Format", "Ghostscript_jll", "InteractiveUtils", "LaTeXStrings", "MacroTools", "Markdown", "OrderedCollections", "Requires"]
git-tree-sha1 = "44f93c47f9cd6c7e431f2f2091fcba8f01cd7e8f"
uuid = "23fbe1c1-3f47-55db-b15f-69d7ec21a316"
version = "0.16.10"

    [deps.Latexify.extensions]
    DataFramesExt = "DataFrames"
    SparseArraysExt = "SparseArrays"
    SymEngineExt = "SymEngine"
    TectonicExt = "tectonic_jll"

    [deps.Latexify.weakdeps]
    DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
    SymEngine = "123dc426-2d89-5057-bbad-38513e3affd8"
    tectonic_jll = "d7dd28d6-a5e6-559c-9131-7eb760cdacc5"

[[deps.LibCURL]]
deps = ["LibCURL_jll", "MozillaCACerts_jll"]
uuid = "b27032c2-a3e7-50c8-80cd-2d36dbcbfd21"
version = "0.6.4"

[[deps.LibCURL_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll", "Zlib_jll", "nghttp2_jll"]
uuid = "deac9b47-8bc7-5906-a0fe-35ac56dc84c0"
version = "8.4.0+0"

[[deps.LibGit2]]
deps = ["Base64", "LibGit2_jll", "NetworkOptions", "Printf", "SHA"]
uuid = "76f85450-5226-5b5a-8eaa-529ad045b433"

[[deps.LibGit2_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll"]
uuid = "e37daf67-58a4-590a-8e99-b0245dd2ffc5"
version = "1.6.4+0"

[[deps.LibSSH2_jll]]
deps = ["Artifacts", "Libdl", "MbedTLS_jll"]
uuid = "29816b5a-b9ab-546f-933c-edad1886dfa8"
version = "1.11.0+1"

[[deps.Libdl]]
uuid = "8f399da3-3557-5675-b5ff-fb832c97cbdb"

[[deps.LinearAlgebra]]
deps = ["Libdl", "OpenBLAS_jll", "libblastrampoline_jll"]
uuid = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"

[[deps.LogExpFunctions]]
deps = ["DocStringExtensions", "IrrationalConstants", "LinearAlgebra"]
git-tree-sha1 = "13ca9e2586b89836fd20cccf56e57e2b9ae7f38f"
uuid = "2ab3a3ac-af41-5b50-aa03-7779005ae688"
version = "0.3.29"

    [deps.LogExpFunctions.extensions]
    LogExpFunctionsChainRulesCoreExt = "ChainRulesCore"
    LogExpFunctionsChangesOfVariablesExt = "ChangesOfVariables"
    LogExpFunctionsInverseFunctionsExt = "InverseFunctions"

    [deps.LogExpFunctions.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    ChangesOfVariables = "9e997f8a-9a97-42d5-a9f1-ce6bfc15e2c0"
    InverseFunctions = "3587e190-3f89-42d0-90ee-14403ec27112"

[[deps.Logging]]
uuid = "56ddb016-857b-54e1-b83d-db4d58db5568"

[[deps.MIMEs]]
git-tree-sha1 = "c64d943587f7187e751162b3b84445bbbd79f691"
uuid = "6c6e2e6c-3030-632d-7369-2d6c69616d65"
version = "1.1.0"

[[deps.MacroTools]]
git-tree-sha1 = "1e0228a030642014fe5cfe68c2c0a818f9e3f522"
uuid = "1914dd2f-81c6-5fcd-8719-6d5c9610ff09"
version = "0.5.16"

[[deps.Markdown]]
deps = ["Base64"]
uuid = "d6f4376e-aef5-505a-96c1-9c027394607a"

[[deps.MbedTLS_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "c8ffd9c3-330d-5841-b78e-0817d7145fa1"
version = "2.28.2+1"

[[deps.Missings]]
deps = ["DataAPI"]
git-tree-sha1 = "ec4f7fbeab05d7747bdf98eb74d130a2a2ed298d"
uuid = "e1d29d7a-bbdc-5cf2-9ac0-f12de2c33e28"
version = "1.2.0"

[[deps.Mmap]]
uuid = "a63ad114-7e13-5084-954f-fe012c677804"

[[deps.MozillaCACerts_jll]]
uuid = "14a3606d-f60d-562e-9121-12d972cd8159"
version = "2023.1.10"

[[deps.NetworkOptions]]
uuid = "ca575930-c2e3-43a9-ace4-1e988b2c1908"
version = "1.2.0"

[[deps.OpenBLAS_jll]]
deps = ["Artifacts", "CompilerSupportLibraries_jll", "Libdl"]
uuid = "4536629a-c528-5b80-bd46-f80d51c5b363"
version = "0.3.23+4"

[[deps.OrderedCollections]]
git-tree-sha1 = "05868e21324cede2207c6f0f466b4bfef6d5e7ee"
uuid = "bac558e1-5e72-5ebc-8fee-abe8a469f55d"
version = "1.8.1"

[[deps.Parameters]]
deps = ["OrderedCollections", "UnPack"]
git-tree-sha1 = "34c0e9ad262e5f7fc75b10a9952ca7692cfc5fbe"
uuid = "d96e819e-fc66-5662-9728-84c9c7592b0a"
version = "0.12.3"

[[deps.Parsers]]
deps = ["Dates", "PrecompileTools", "UUIDs"]
git-tree-sha1 = "7d2f8f21da5db6a806faf7b9b292296da42b2810"
uuid = "69de0a69-1ddd-5017-9359-2bf0b02dc9f0"
version = "2.8.3"

[[deps.Pkg]]
deps = ["Artifacts", "Dates", "Downloads", "FileWatching", "LibGit2", "Libdl", "Logging", "Markdown", "Printf", "REPL", "Random", "SHA", "Serialization", "TOML", "Tar", "UUIDs", "p7zip_jll"]
uuid = "44cfe95a-1eb2-52ea-b672-e2afdf69b78f"
version = "1.10.0"

[[deps.PlotlyBase]]
deps = ["ColorSchemes", "Colors", "Dates", "DelimitedFiles", "DocStringExtensions", "JSON", "LaTeXStrings", "Logging", "Parameters", "Pkg", "REPL", "Requires", "Statistics", "UUIDs"]
git-tree-sha1 = "6256ab3ee24ef079b3afa310593817e069925eeb"
uuid = "a03496cd-edff-5a9b-9e67-9cda94a718b5"
version = "0.8.23"

    [deps.PlotlyBase.extensions]
    DataFramesExt = "DataFrames"
    DistributionsExt = "Distributions"
    IJuliaExt = "IJulia"
    JSON3Ext = "JSON3"

    [deps.PlotlyBase.weakdeps]
    DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
    Distributions = "31c24e10-a181-5473-b8eb-7969acd0382f"
    IJulia = "7073ff75-c697-5162-941a-fcdaad2a7d2a"
    JSON3 = "0f8b85d8-7281-11e9-16c2-39a750bddbf1"

[[deps.PlotlyKaleido]]
deps = ["Artifacts", "Base64", "JSON", "Kaleido_jll"]
git-tree-sha1 = "c7a270d11881c0709052f8e3def3d60d551e99f3"
uuid = "f2990250-8cf9-495f-b13a-cce12b45703c"
version = "2.3.1"

[[deps.PlutoPlotly]]
deps = ["AbstractPlutoDingetjes", "Artifacts", "ColorSchemes", "Colors", "Dates", "Downloads", "HypertextLiteral", "InteractiveUtils", "LaTeXStrings", "Markdown", "Pkg", "PlotlyBase", "PrecompileTools", "Reexport", "ScopedValues", "Scratch", "TOML"]
git-tree-sha1 = "8acd04abc9a636ef57004f4c2e6f3f6ed4611099"
uuid = "8e989ff0-3d88-8e9f-f020-2b208a939ff0"
version = "0.6.5"

    [deps.PlutoPlotly.extensions]
    PlotlyKaleidoExt = "PlotlyKaleido"
    UnitfulExt = "Unitful"

    [deps.PlutoPlotly.weakdeps]
    PlotlyKaleido = "f2990250-8cf9-495f-b13a-cce12b45703c"
    Unitful = "1986cc42-f94f-5a68-af5c-568840ba703d"

[[deps.PlutoUI]]
deps = ["AbstractPlutoDingetjes", "Base64", "ColorTypes", "Dates", "Downloads", "FixedPointNumbers", "Hyperscript", "HypertextLiteral", "IOCapture", "InteractiveUtils", "Logging", "MIMEs", "Markdown", "Random", "Reexport", "URIs", "UUIDs"]
git-tree-sha1 = "3ac7038a98ef6977d44adeadc73cc6f596c08109"
uuid = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
version = "0.7.79"

[[deps.PooledArrays]]
deps = ["DataAPI", "Future"]
git-tree-sha1 = "36d8b4b899628fb92c2749eb488d884a926614d3"
uuid = "2dfb63ee-cc39-5dd5-95bd-886bf059d720"
version = "1.4.3"

[[deps.PrecompileTools]]
deps = ["Preferences"]
git-tree-sha1 = "5aa36f7049a63a1528fe8f7c3f2113413ffd4e1f"
uuid = "aea7be01-6a6a-4083-8856-8a6e6704d82a"
version = "1.2.1"

[[deps.Preferences]]
deps = ["TOML"]
git-tree-sha1 = "522f093a29b31a93e34eaea17ba055d850edea28"
uuid = "21216c6a-2e73-6563-6e65-726566657250"
version = "1.5.1"

[[deps.PrettyTables]]
deps = ["Crayons", "LaTeXStrings", "Markdown", "PrecompileTools", "Printf", "REPL", "Reexport", "StringManipulation", "Tables"]
git-tree-sha1 = "c5a07210bd060d6a8491b0ccdee2fa0235fc00bf"
uuid = "08abe8d2-0d0c-5749-adfa-8a2ac140af0d"
version = "3.1.2"

[[deps.Printf]]
deps = ["Unicode"]
uuid = "de0858da-6303-5e67-8744-51eddeeeb8d7"

[[deps.PtrArrays]]
git-tree-sha1 = "1d36ef11a9aaf1e8b74dacc6a731dd1de8fd493d"
uuid = "43287f4e-b6f4-7ad1-bb20-aadabca52c3d"
version = "1.3.0"

[[deps.REPL]]
deps = ["InteractiveUtils", "Markdown", "Sockets", "Unicode"]
uuid = "3fa0cd96-eef1-5676-8a61-b3b8758bbffb"

[[deps.Random]]
deps = ["SHA"]
uuid = "9a3f8284-a2c9-5f02-9a11-845980a1fd5c"

[[deps.Reexport]]
git-tree-sha1 = "45e428421666073eab6f2da5c9d310d99bb12f9b"
uuid = "189a3867-3050-52da-a836-e630ba90ab69"
version = "1.2.2"

[[deps.Requires]]
deps = ["UUIDs"]
git-tree-sha1 = "62389eeff14780bfe55195b7204c0d8738436d64"
uuid = "ae029012-a4dd-5104-9daa-d747884805df"
version = "1.3.1"

[[deps.SHA]]
uuid = "ea8e919c-243c-51af-8825-aaa63cd721ce"
version = "0.7.0"

[[deps.ScopedValues]]
deps = ["HashArrayMappedTries", "Logging"]
git-tree-sha1 = "c3b2323466378a2ba15bea4b2f73b081e022f473"
uuid = "7e506255-f358-4e82-b7e4-beb19740aa63"
version = "1.5.0"

[[deps.Scratch]]
deps = ["Dates"]
git-tree-sha1 = "9b81b8393e50b7d4e6d0a9f14e192294d3b7c109"
uuid = "6c6a2e73-6563-6170-7368-637461726353"
version = "1.3.0"

[[deps.SentinelArrays]]
deps = ["Dates", "Random"]
git-tree-sha1 = "ebe7e59b37c400f694f52b58c93d26201387da70"
uuid = "91c51154-3ec4-41a3-a24f-3f23e20d615c"
version = "1.4.9"

[[deps.Serialization]]
uuid = "9e88b42a-f829-5b0c-bbe9-9e923198166b"

[[deps.ShiftedArrays]]
git-tree-sha1 = "503688b59397b3307443af35cd953a13e8005c16"
uuid = "1277b4bf-5013-50f5-be3d-901d8477a67a"
version = "2.0.0"

[[deps.Sockets]]
uuid = "6462fe0b-24de-5631-8697-dd941f90decc"

[[deps.SortingAlgorithms]]
deps = ["DataStructures"]
git-tree-sha1 = "64d974c2e6fdf07f8155b5b2ca2ffa9069b608d9"
uuid = "a2af1166-a08f-5f64-846c-94a0d3cef48c"
version = "1.2.2"

[[deps.SparseArrays]]
deps = ["Libdl", "LinearAlgebra", "Random", "Serialization", "SuiteSparse_jll"]
uuid = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
version = "1.10.0"

[[deps.Statistics]]
deps = ["LinearAlgebra", "SparseArrays"]
uuid = "10745b16-79ce-11e8-11f9-7d13ad32a3b2"
version = "1.10.0"

[[deps.StatsAPI]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "178ed29fd5b2a2cfc3bd31c13375ae925623ff36"
uuid = "82ae8749-77ed-4fe6-ae5f-f523153014b0"
version = "1.8.0"

[[deps.StatsBase]]
deps = ["AliasTables", "DataAPI", "DataStructures", "IrrationalConstants", "LinearAlgebra", "LogExpFunctions", "Missings", "Printf", "Random", "SortingAlgorithms", "SparseArrays", "Statistics", "StatsAPI"]
git-tree-sha1 = "aceda6f4e598d331548e04cc6b2124a6148138e3"
uuid = "2913bbd2-ae8a-5f71-8c99-4fb6c76f3a91"
version = "0.34.10"

[[deps.StringManipulation]]
deps = ["PrecompileTools"]
git-tree-sha1 = "a3c1536470bf8c5e02096ad4853606d7c8f62721"
uuid = "892a3eda-7b42-436c-8928-eab12a02cf0e"
version = "0.4.2"

[[deps.StructUtils]]
deps = ["Dates", "UUIDs"]
git-tree-sha1 = "9297459be9e338e546f5c4bedb59b3b5674da7f1"
uuid = "ec057cc2-7a8d-4b58-b3b3-92acb9f63b42"
version = "2.6.2"

    [deps.StructUtils.extensions]
    StructUtilsMeasurementsExt = ["Measurements"]
    StructUtilsTablesExt = ["Tables"]

    [deps.StructUtils.weakdeps]
    Measurements = "eff96d63-e80a-5855-80a2-b1b0885c5ab7"
    Tables = "bd369af6-aec1-5ad0-b16a-f7cc5008161c"

[[deps.SuiteSparse_jll]]
deps = ["Artifacts", "Libdl", "libblastrampoline_jll"]
uuid = "bea87d4a-7f5b-5778-9afe-8cc45184846c"
version = "7.2.1+1"

[[deps.TOML]]
deps = ["Dates"]
uuid = "fa267f1f-6049-4f14-aa54-33bafae1ed76"
version = "1.0.3"

[[deps.TableTraits]]
deps = ["IteratorInterfaceExtensions"]
git-tree-sha1 = "c06b2f539df1c6efa794486abfb6ed2022561a39"
uuid = "3783bdb8-4a98-5b6b-af9a-565f29a5fe9c"
version = "1.0.1"

[[deps.Tables]]
deps = ["DataAPI", "DataValueInterfaces", "IteratorInterfaceExtensions", "OrderedCollections", "TableTraits"]
git-tree-sha1 = "f2c1efbc8f3a609aadf318094f8fc5204bdaf344"
uuid = "bd369af6-aec1-5ad0-b16a-f7cc5008161c"
version = "1.12.1"

[[deps.Tar]]
deps = ["ArgTools", "SHA"]
uuid = "a4e569a6-e804-4fa4-b0f3-eef7a1d5b13e"
version = "1.10.0"

[[deps.TensorCore]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "1feb45f88d133a655e001435632f019a9a1bcdb6"
uuid = "62fd8b95-f654-4bbd-a8a5-9c27f68ccd50"
version = "0.1.1"

[[deps.Test]]
deps = ["InteractiveUtils", "Logging", "Random", "Serialization"]
uuid = "8dfed614-e22c-5e08-85e1-65c5234f0b40"

[[deps.TranscodingStreams]]
git-tree-sha1 = "0c45878dcfdcfa8480052b6ab162cdd138781742"
uuid = "3bb67fe8-82b1-5028-8e26-92a6c54297fa"
version = "0.11.3"

[[deps.Tricks]]
git-tree-sha1 = "311349fd1c93a31f783f977a71e8b062a57d4101"
uuid = "410a4b4d-49e4-4fbc-ab6d-cb71b17b3775"
version = "0.1.13"

[[deps.URIs]]
git-tree-sha1 = "bef26fb046d031353ef97a82e3fdb6afe7f21b1a"
uuid = "5c2747f8-b7ea-4ff2-ba2e-563bfd36b1d4"
version = "1.6.1"

[[deps.UUIDs]]
deps = ["Random", "SHA"]
uuid = "cf7118a7-6976-5b1a-9a39-7adc72f591a4"

[[deps.UnPack]]
git-tree-sha1 = "387c1f73762231e86e0c9c5443ce3b4a0a9a0c2b"
uuid = "3a884ed6-31ef-47d7-9d2a-63182c4928ed"
version = "1.0.2"

[[deps.Unicode]]
uuid = "4ec0a83e-493e-50e2-b9ac-8f72acf5a8f5"

[[deps.WeakRefStrings]]
deps = ["DataAPI", "InlineStrings", "Parsers"]
git-tree-sha1 = "b1be2855ed9ed8eac54e5caff2afcdb442d52c23"
uuid = "ea10d353-3f73-51f8-a26c-33c1cb351aa5"
version = "1.4.2"

[[deps.WorkerUtilities]]
git-tree-sha1 = "cd1659ba0d57b71a464a29e64dbc67cfe83d54e7"
uuid = "76eceee3-57b5-4d4a-8e66-0e911cebbf60"
version = "1.6.1"

[[deps.Zlib_jll]]
deps = ["Libdl"]
uuid = "83775a58-1f1d-513f-b197-d71354ab007a"
version = "1.2.13+1"

[[deps.libblastrampoline_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850b90-86db-534c-a0d3-1478176c7d93"
version = "5.11.0+0"

[[deps.nghttp2_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850ede-7688-5339-a07c-302acd2aaf8d"
version = "1.52.0+1"

[[deps.p7zip_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "3f19e933-33d8-53b3-aaab-bd5110c3b7a0"
version = "17.4.0+2"
"""

# ╔═╡ Cell order:
# ╟─9d1ca4cc-688c-4693-9311-25b33feca957
# ╟─0b869e70-3717-11ec-202c-6b94dac7daf3
# ╟─b326715f-c37f-44d2-92f9-6264368a18b2
# ╟─4fe446fc-10cf-4439-ae07-e0dad3a54de0
# ╟─554b2cc5-99e1-4eda-b080-67ab2eddac16
# ╠═b72215bb-8e3a-46e2-8817-271befe1178c
# ╟─4d91f31e-06ce-4124-8a10-6619010c9476
# ╟─5e8a5e83-9cc0-4bb6-86d2-4ec5f3d19b99
# ╟─0d7ff3f1-7e34-4317-857b-c212d4f5989c
# ╟─99cb7dc3-f7d5-4cb5-9301-99ddc860b52c
# ╟─b5106532-729d-46b5-b0fe-886c4349ee11
# ╟─c81b7a4a-21fb-4643-ba05-7b744d29e70a
# ╟─22f0e100-55e6-4849-91cb-c22fc334b5cc
# ╟─63610272-6b65-41f6-a71d-eb3c1327e4b6
# ╟─30f88ccb-735b-4842-8598-751d985a7ff0
# ╟─116cc0c2-2674-4c05-948c-bc3f1915e30c
# ╟─eaf08952-9ce9-4c80-a708-52487825e13f
# ╟─b7113a0b-6370-461f-ad47-d5e16c81c6cd
# ╟─b8b07199-9deb-4108-95ea-12a2bcd5069c
# ╟─6d03748d-00e0-4c18-a8bc-d432a72714d2
# ╟─3c77e1e2-b5f2-4440-b4b1-08149e7d453b
# ╟─47ab9b42-f7cf-4226-8287-f1efe07f73c8
# ╟─6d85d1f3-e1e4-4a80-89b7-1071b20de7aa
# ╟─d816ec4a-6478-4508-b7e3-22466a75c5cb
# ╟─95453c1f-1a35-4658-a9fc-60e8bbdfbc3d
# ╠═d4fc54f0-c37a-44f1-84f7-fe21c8e2862a
# ╠═9fa258be-77bf-4ac2-80fc-a6711977ec28
# ╟─990bb081-5591-4b13-9e5a-e93efc3afc76
# ╠═59b72098-9005-4f18-a196-a2da0bf50a87
# ╠═ace171a3-42b5-4798-a471-873b5a2e1cb7
# ╠═4c13cf93-f7d3-4d4c-a4a8-5f7259984a3e
# ╠═c7841346-c3ab-4ca8-9371-b0e6657274e5
# ╟─8bee12a0-7109-4bd0-9681-8b447693984f
# ╟─5d2d0792-eb31-4e4e-a86a-22d46bd9ee6f
# ╟─2a948df7-6bc0-4ab0-b736-49497b4f3d4c
# ╠═d33ca297-c94a-4ba6-970b-df8bca54d428
# ╟─a9cfeb52-b360-4661-8c59-b6882d04469b
# ╟─f2956bed-9534-4c57-bfaa-fbef3c4af810
# ╠═65ec39b9-0147-4ab2-a18b-ec330af5cf2f
# ╟─1dac18c0-836c-4ee5-a5e4-bd120ce46bc5
# ╠═feaab8f3-93c5-4f92-a236-2532a388cd51
# ╟─08d213c0-9be4-48a2-9463-2534a6a5d7ec
# ╠═3ddf9a66-f391-4e30-9168-9dc2e27a49b9
# ╟─e9a49103-860e-426e-84c7-f41fb3fab9b4
# ╠═9a4394a5-928a-4b1a-b40a-3e50e34a92ba
# ╟─121cb8d7-b1e5-454d-8e73-26b4d5fff4ad
# ╠═ab172c4a-5e0b-404b-a522-4f10e1ed4f40
# ╠═8f412cf8-bcb6-4cb5-9d59-88e7e5f704d6
# ╠═829e2519-69f1-4ea0-8f68-6a0ba22552b7
# ╠═a5a1cbe5-cda6-4017-8b7f-3bce54951dc8
# ╠═a0541145-d76c-40bb-8a51-17f1d42ead0a
# ╠═495f6688-5a97-4747-9dd2-45009b238cc0
# ╟─f1d69f4e-e1f3-490e-9b0a-9d75204fcaac
# ╟─60bab72a-81ac-40a5-b7de-ed822e012f6f
# ╟─e5c4408d-0174-4f39-8a58-121502c0f193
# ╟─db133e97-b597-48f6-a232-180ae9cecbab
# ╟─6aa9df0f-cf04-4d72-906a-c1c1badf449b
# ╟─ffe77616-e884-40b1-8d1e-32b5573c547f
# ╟─9b09c201-ffc2-4d10-9958-c9523d60ccae
# ╠═5d515df6-6487-4a0c-9e05-8df53631c0d5
# ╠═ec20be10-13c3-45a0-ba9c-5b986740f658
# ╠═7b3ec0c9-0f2e-406e-a76d-bbea872c2b65
# ╠═e0b538a5-5cd8-4a89-a8de-b6e9f339add9
# ╠═650e8dc7-31b7-4a97-b4fa-24e27c9da1ca
# ╟─24b95a91-d2f8-435f-80a8-764f1534c3eb
# ╟─52c76dc2-146c-4178-9aa3-d132a0551168
# ╟─fff5ce25-8546-43b5-9d82-ec3c560105dd
# ╟─b48ad882-52b8-4809-93ae-ad22b5705e3c
# ╟─c3ea937d-f8db-4e3d-b12d-50957efb3040
# ╟─652c8219-1530-47b1-adcb-17696171bab6
# ╠═75ca182d-df13-4248-9d4c-5f572fc89637
# ╟─bf61c3c3-bba6-4417-8a11-3603c6c20378
# ╠═8fcbd455-bd06-4fe7-96cb-a15de9e80d79
# ╟─4847b31b-1d00-44d8-b1da-745559807a9a
# ╟─dafa708f-e667-4c1b-ac44-5c639895f60c
# ╠═af7929eb-ab80-4275-bd45-09c95dc22079
# ╟─e63efa59-a6f2-41c0-b5a1-a82630636f65
# ╟─bfb7998b-4445-4023-af98-ae0790af0b8c
# ╠═00b75cd1-8a92-4523-9ff2-d180bf25e869
# ╟─3d2c9595-3cde-4279-a7ee-2efa608998f6
# ╟─268cab98-86dc-4070-a285-d00950a089b1
# ╟─cee5cafb-07a7-4804-8589-ddae72a439a8
# ╟─7f962bfb-1773-4179-882e-d17686b4ea50
# ╟─cacc9719-25a9-4b03-b92a-07bbca9e0acd
# ╠═93979e82-668f-416b-bec4-f6cea410a83d
# ╟─82c4d622-febe-49ca-b827-5ae443f3d561
# ╟─9b07963e-a523-490d-aedd-c9d59b6dff84
# ╟─625cf0da-c085-4d1e-98ff-5e5091589923
# ╟─40de1d39-effa-460b-8ccd-85d5be501a63
# ╠═8892c48d-5647-4ecf-9edb-51370332d171
# ╟─5ec7390c-2383-4033-9089-1c051b806912
# ╟─095691a9-2b67-439e-a19f-61312496b3b5
# ╟─e3772aa9-c1f1-4989-b0f6-50c12a6fe89c
# ╟─09b9db62-c44e-4bdb-964f-4c95c73d417a
# ╠═6857086e-c1d2-49e4-b54d-2b8267b1664f
# ╟─e77e17bb-4275-4e59-9478-79231bac359c
# ╟─e6b89599-8a85-455e-a96b-9d3ddeeb3000
# ╟─13cb0884-742b-4444-a84f-1290ebf2b90e
# ╟─b641186e-e584-4f21-967d-48bc39f25569
# ╟─585f0c52-1344-4d60-89fd-a59bbbf40ed9
# ╟─9aef5e9f-bfcb-4489-8eef-954ad4dfdc0a
# ╟─1e532543-f979-4d8a-81db-a2cfaf4018e9
# ╟─800793c6-201a-4d09-b704-859fb90cc138
# ╟─dd1a41d5-de1b-4a0f-82fa-96ce46972145
# ╟─2cdee962-2074-4f04-905a-6c2109747d4f
# ╠═bf81dbd0-6181-4cd2-aba1-31a72c4bce0b
# ╟─7b542867-7e3d-430e-afe4-c70726fb833c
# ╠═2f65205b-5410-435a-a771-6c5d746b2472
# ╟─55962895-c38a-474b-b737-42512b891986
# ╟─f5aa880c-06f8-4a26-83ac-097db126eabc
# ╟─f3e3c449-1642-419d-9fd3-d7f7297f812f
# ╠═66936f30-7c10-47b2-8ab3-47c8d8a40e59
# ╟─da40f057-fc5c-4f82-be14-1d1cf9a00204
# ╠═12586f44-f402-4c28-ba7d-00b418fbeaf0
# ╟─84038536-234c-46e9-ae0c-cc755d92a9d9
# ╠═3cbefd1b-8892-43a1-affb-117613d2dadf
# ╟─330f9526-66f9-4cfc-a3f0-61d58af7b9b5
# ╟─66779d90-b2ab-4691-9e2e-b7333729e708
# ╟─ad23ba06-2cb7-4ec9-a1c5-32f78b2ee145
# ╟─be339830-24b1-4011-b626-589b0158399d
# ╟─659da403-d7d0-47c7-bea1-237c4b09e93d
# ╠═58d0be12-843e-46cb-8c74-c8c3de4a45db
# ╠═8f1471b1-0471-4890-9469-41f1369bc0ff
# ╟─eb76ac98-8640-4eea-a616-3c867229fe60
# ╠═2215a118-0a8d-454a-9f48-bd23208c0d81
# ╟─9b4dd09d-b1ea-4041-a097-24e8e10737ad
# ╟─8e7e6b3a-a2be-4e20-b88c-1f4dfe00be43
# ╠═9871cd1b-bbba-40d9-96f5-c388d20cc4c6
# ╟─2f8087f3-0bbe-4715-9cf3-feb02ab3da95
# ╟─138cf359-5a5e-4417-a6d9-d2b86f6d5012
# ╠═539af3d5-9430-47bc-8df0-fd8d5f575b59
# ╟─bd8d669a-ee76-447d-9739-6428251c92ee
# ╟─edbd2b1c-1452-41a6-962c-3434bd420020
# ╟─d9672a6f-97af-43d1-9a30-50495204fec8
# ╟─f6283c43-c912-430d-a63d-bba0df690cd4
# ╟─c59c18ff-f4a4-4b28-8e5a-2c7a03e86d83
# ╠═021ee205-8123-4158-9c75-f1cc7f2ab6f2
# ╟─00000000-0000-0000-0000-000000000001
# ╟─00000000-0000-0000-0000-000000000002
